-- Author: U_BMP
-- Group: https://vk.com/biomodprod_utilit_fs
-- Date: 29.12.2025

BMP_ExhaustPlusSpec = {}
BMP_ExhaustPlusSpec.modDirectory = g_currentModDirectory or ""
BMP_ExhaustPlusSpec.modName = g_currentModName or "BMP_ExhaustPlus"

-- -----------------------------
-- SETTINGS
-- -----------------------------
BMP_ExhaustPlusSpec.DEBUG = true

BMP_ExhaustPlusSpec.extraParticlesEnable = true
BMP_ExhaustPlusSpec.extraEmitterShapeI3D = "particles/exhaustEmitShape.i3d"

-- ---------------------------------------------------------
-- PARTICLE PRESETS (ВАРИАНТЫ ПАРТИКЛОВ)
-- ---------------------------------------------------------
BMP_ExhaustPlusSpec.DEFAULT_PRESET_KEY = "default"

BMP_ExhaustPlusSpec.PARTICLE_PRESETS = {
    default = {
        layerA_enable       = true,
        layerA_particlesI3D = "particles/variant1/exhaust.i3d",
        layerB_enable       = true,
        layerB_particlesI3D = "particles/variant1/exhaust2.i3d",
        layerC_enable       = true,
        layerC_particlesI3D = "particles/variant1/exhaust3.i3d",
        layerD_enable       = true,
        layerD_particlesI3D = "particles/variant1/exhaust4.i3d",
    },

    tractor = {
        layerA_enable       = true,
        layerA_particlesI3D = "particles/variant4/smoke.i3d",
        layerB_enable       = true,
        layerB_particlesI3D = "particles/variant4/smoke2.i3d",
        layerC_enable       = true,
        layerC_particlesI3D = "particles/variant4/smoke3.i3d",
        layerD_enable       = true,
        layerD_particlesI3D = "particles/variant4/smoke4.i3d",
    },

    tractor2 = {
        layerA_enable       = true,
        layerA_particlesI3D = "particles/variant2/smoke.i3d",
        layerB_enable       = true,
        layerB_particlesI3D = "particles/variant2/smoke2.i3d",
        layerC_enable       = true,
        layerC_particlesI3D = "particles/variant2/smoke3.i3d",
        layerD_enable       = true,
        layerD_particlesI3D = "particles/variant2/smoke4.i3d",
    },

    tractorPuff = {
        layerA_enable       = true,
        layerA_particlesI3D = "particles/variant4/smoke.i3d",
        layerB_enable       = true,
        layerB_particlesI3D = "particles/variant4/smoke2.i3d",
        layerC_enable       = true,
        layerC_particlesI3D = "particles/variant4/smoke3.i3d",
        layerD_enable       = true,
        layerD_particlesI3D = "particles/variant4/smoke4.i3d",

        -- стартовый "пых" при заводе
        layerStart_enable       = true,
        layerStart_particlesI3D = "particles/variant4/smoke3.i3d",
    },

    carFillable = {
        layerA_enable       = true,
        layerA_particlesI3D = "particles/variant3/smoke.i3d",
        layerB_enable       = true,
        layerB_particlesI3D = "particles/variant3/smoke2.i3d",
        layerC_enable       = true,
        layerC_particlesI3D = "particles/variant3/smoke3.i3d",
        layerD_enable       = true,
        layerD_particlesI3D = "particles/variant3/smoke4.i3d",
    },

    combineDrivable = {
        layerA_enable       = true,
        layerA_particlesI3D = "particles/variant3/smoke.i3d",
        layerB_enable       = true,
        layerB_particlesI3D = "particles/variant3/smoke2.i3d",
        layerC_enable       = true,
        layerC_particlesI3D = "particles/variant3/smoke3.i3d",
        layerD_enable       = true,
        layerD_particlesI3D = "particles/variant3/smoke4.i3d",
    }
}

BMP_ExhaustPlusSpec.TYPE_TO_PRESET = {
    tractor          = "tractor",
    tractor2         = "tractor2",
    combineDrivable  = "combineDrivable",
    baseDrivable     = "baseDrivable",
    carFillable      = "carFillable",
    tractorPuff      = "tractorPuff",
}

BMP_ExhaustPlusSpec.START_PUFF_DURATION_MS = 438  -- сколько длится "пых"
BMP_ExhaustPlusSpec.START_PUFF_DELAY_MS    = 355  -- задержка перед "пых" после запуска двигателя

BMP_ExhaustPlusSpec.particleNodeIndex = "0"
BMP_ExhaustPlusSpec.particleClipDist  = 150

BMP_ExhaustPlusSpec.offsetX = 0
BMP_ExhaustPlusSpec.offsetY = 0.06
BMP_ExhaustPlusSpec.offsetZ = 0

-- Скрывать ли ванильный эффект
BMP_ExhaustPlusSpec.HIDE_VANILLA_EXHAUST    = true
BMP_ExhaustPlusSpec.ACTIVE_SCAN_INTERVAL_MS = 200

-- -----------------------------
-- Образование дыма при НАГРУЗКЕ + оборотах в минуту
-- -----------------------------
BMP_ExhaustPlusSpec.TUNE_INTERVAL_MS = 75
BMP_ExhaustPlusSpec.TUNE_EPS         = 0.02

BMP_ExhaustPlusSpec.LOAD_EXP = 1.35
BMP_ExhaustPlusSpec.RPM_EXP  = 1.01

BMP_ExhaustPlusSpec.RPM_GATE_MIN = 0.10
BMP_ExhaustPlusSpec.RPM_GATE_MAX = 0.22

BMP_ExhaustPlusSpec.MIN_INTENSITY_TO_EMIT  = 0.01
BMP_ExhaustPlusSpec.A_IDLE_INTENSITY_FLOOR = 0.07

-- =========================================================
-- ИСКЛЮЧИТЕЛЬНЫЕ ПОРОГОВЫЕ ЗНАЧЕНИЯ (интенсивность 0..1)
-- =========================================================
BMP_ExhaustPlusSpec.THR_AB_ON  = 0.10
BMP_ExhaustPlusSpec.THR_BC_ON  = 0.26
BMP_ExhaustPlusSpec.THR_CD_ON  = 0.60

BMP_ExhaustPlusSpec.THR_AB_OFF = 0.07
BMP_ExhaustPlusSpec.THR_BC_OFF = 0.20
BMP_ExhaustPlusSpec.THR_CD_OFF = 0.52

-- ---------------------------------------------------------
-- Визуальная стабилизация
-- ---------------------------------------------------------
BMP_ExhaustPlusSpec.RESET_ON_RAMP_DELTA = 0.18
BMP_ExhaustPlusSpec.RESET_COOLDOWN_MS   = 650

-- =========================================================
-- НАСТРАИВАЕМЫЕ КРИВЫЕ (по слоям)
-- =========================================================
BMP_ExhaustPlusSpec.A_EMIT_SCALE_MIN = 0.04
BMP_ExhaustPlusSpec.A_EMIT_SCALE_MAX = 0.34
BMP_ExhaustPlusSpec.A_SPEED_MUL_MIN  = 0.95
BMP_ExhaustPlusSpec.A_SPEED_MUL_MAX  = 1.15

BMP_ExhaustPlusSpec.B_EMIT_SCALE_MIN = 0.12
BMP_ExhaustPlusSpec.B_EMIT_SCALE_MAX = 2.10
BMP_ExhaustPlusSpec.B_SPEED_MUL_MIN  = 1.05
BMP_ExhaustPlusSpec.B_SPEED_MUL_MAX  = 1.25

BMP_ExhaustPlusSpec.C_EMIT_SCALE_MIN = 0.20
BMP_ExhaustPlusSpec.C_EMIT_SCALE_MAX = 3.84
BMP_ExhaustPlusSpec.C_SPEED_MUL_MIN  = 1.15
BMP_ExhaustPlusSpec.C_SPEED_MUL_MAX  = 1.30

BMP_ExhaustPlusSpec.D_EMIT_SCALE_MIN = 0.32
BMP_ExhaustPlusSpec.D_EMIT_SCALE_MAX = 8.02
BMP_ExhaustPlusSpec.D_SPEED_MUL_MIN  = 1.28
BMP_ExhaustPlusSpec.D_SPEED_MUL_MAX  = 1.45

-- =========================================================
-- HUD STATUS (НИЗ ПО ЦЕНТРУ)
-- =========================================================
BMP_ExhaustPlusSpec.HUD_STATUS_ENABLE      = true
BMP_ExhaustPlusSpec.HUD_STATUS_DURATION_MS = 2200
BMP_ExhaustPlusSpec.HUD_STATUS_Y           = 0.020
BMP_ExhaustPlusSpec.HUD_STATUS_TEXT_SIZE   = 0.018

-- =========================================================

local function log(fmt, ...)
    if BMP_ExhaustPlusSpec.DEBUG then
        print(("[BMP_ExhaustPlus] " .. fmt):format(...))
    end
end

local function mpSafeDelete(node)
    if node ~= nil and node ~= 0 and entityExists(node) then
        delete(node)
    end
end

local function clamp(x, a, b)
    if x == nil then return a end
    if x < a then return a end
    if x > b then return b end
    return x
end

local function lerp(a, b, t)
    return a + (b - a) * t
end

local function smoothstep(edge0, edge1, x)
    if edge0 == edge1 then
        return x >= edge1 and 1 or 0
    end
    local t = clamp((x - edge0) / (edge1 - edge0), 0, 1)
    return t * t * (3 - 2 * t)
end

-- =========================================================
-- HUD helpers
-- =========================================================
local function getHudStatusText(vehicle)
    local spec = vehicle ~= nil and vehicle.spec_bmpExhaustPlus or nil
    if spec == nil then
        return nil
    end

    local customOn = (spec.userEnabled ~= false)
    local presetKey = spec.userPresetKey or spec._presetKey or (BMP_ExhaustPlusSpec.DEFAULT_PRESET_KEY or "default")

    local vanillaOn = true
    if customOn and (BMP_ExhaustPlusSpec.HIDE_VANILLA_EXHAUST == true) then
        vanillaOn = false
    end

    local tCustom = customOn and "ON" or "OFF"
    local tVan = vanillaOn and "ON" or "OFF"

    return string.format("ExhaustPlus | Custom: %s | Preset: %s | Vanilla: %s", tCustom, tostring(presetKey), tVan)
end

function BMP_ExhaustPlusSpec.pushHudStatus(vehicle)
    if vehicle == nil or not vehicle.isClient or BMP_ExhaustPlusSpec.HUD_STATUS_ENABLE ~= true then
        return
    end
    local spec = vehicle.spec_bmpExhaustPlus
    if spec == nil then
        return
    end

    spec._hudStatusText = getHudStatusText(vehicle)
    spec._hudStatusMs   = BMP_ExhaustPlusSpec.HUD_STATUS_DURATION_MS or 2000
end

local function tickHud(spec, dt)
    if spec == nil then return end
    if (spec._hudStatusMs or 0) > 0 then
        spec._hudStatusMs = math.max((spec._hudStatusMs or 0) - dt, 0)
        if (spec._hudStatusMs or 0) <= 0 then
            spec._hudStatusText = nil
        end
    end
end

local function drawHud(spec)
    if spec == nil then return end
    if (spec._hudStatusMs or 0) <= 0 then return end

    local text = spec._hudStatusText
    if text == nil or text == "" then return end

    local size = BMP_ExhaustPlusSpec.HUD_STATUS_TEXT_SIZE or 0.018
    local y    = BMP_ExhaustPlusSpec.HUD_STATUS_Y or 0.02

    pcall(setTextBold, true)
    pcall(setTextColor, 1, 1, 1, 1)

    if RenderText ~= nil and RenderText.ALIGN_CENTER ~= nil and setTextAlignment ~= nil then
        setTextAlignment(RenderText.ALIGN_CENTER)
        renderText(0.5, y, size, text)
        setTextAlignment(RenderText.ALIGN_LEFT)
    else
        local w = 0
        if getTextWidth ~= nil then
            w = getTextWidth(size, text) or 0
        end
        renderText(0.5 - w * 0.5, y, size, text)
    end

    pcall(setTextBold, false)
end

-- =========================================================

BMP_ExhaustPlusSpec._typeChainCache = BMP_ExhaustPlusSpec._typeChainCache or {}

local function getShortTypeName(typeName)
    if typeName == nil then return nil end
    local s = tostring(typeName)
    local last = s:match("([^%.]+)$")
    return last or s
end

local function pushUnique(arr, seen, v)
    if v == nil then return end
    v = tostring(v)
    if v == "" then return end
    if not seen[v] then
        table.insert(arr, v)
        seen[v] = true
    end
end

local function getParentNameFromTypeDesc(td)
    if td == nil then return nil end

    local p = td.parentName or td.parentTypeName or td.parentType

    if (p == nil or p == "") and td.parent ~= nil then
        if type(td.parent) == "string" then
            p = td.parent
        elseif type(td.parent) == "table" then
            p = td.parent.name or td.parent.typeName or td.parent.parentName or td.parent.parentTypeName
        end
    end

    if type(p) == "table" then
        p = p.name or p.typeName or p.parentName or p.parentTypeName
    end

    if p ~= nil then
        p = tostring(p)
        if p == "" then p = nil end
    end
    return p
end

local function getNameFromTypeDesc(td)
    if td == nil then return nil end
    local n = td.name or td.typeName or td.type or td.xmlName
    if n ~= nil then
        n = tostring(n)
        if n == "" then n = nil end
    end
    return n
end

local function buildChainFromTypeDesc(td, chain, seen)
    if td == nil then return end

    local cur = td
    local guard = 0
    while cur ~= nil and guard < 64 do
        local n = getNameFromTypeDesc(cur)
        pushUnique(chain, seen, n)
        pushUnique(chain, seen, getShortTypeName(n))

        local pName = getParentNameFromTypeDesc(cur)
        if pName == nil then
            break
        end

        local nextTd = nil
        if type(cur.parentType) == "table" then
            nextTd = cur.parentType
        elseif type(cur.parent) == "table" then
            nextTd = cur.parent
        end

        if nextTd ~= nil then
            cur = nextTd
        else
            pushUnique(chain, seen, pName)
            pushUnique(chain, seen, getShortTypeName(pName))
            break
        end

        guard = guard + 1
    end
end

local function vtmLookupAny(vtm, name)
    if vtm == nil or name == nil or name == "" then
        return nil
    end

    if vtm.getVehicleTypeByName ~= nil then
        local vt = vtm:getVehicleTypeByName(name)
        if vt ~= nil then return vt end
    end
    if vtm.getTypeByName ~= nil then
        local vt = vtm:getTypeByName(name)
        if vt ~= nil then return vt end
    end
    if vtm.getVehicleType ~= nil then
        local vt = vtm:getVehicleType(name)
        if vt ~= nil then return vt end
    end

    local tables = {
        vtm.vehicleTypes,
        vtm.types,
        vtm.typeNameToDesc,
        vtm.typeNameToType,
        vtm.nameToType,
        vtm.nameToDesc
    }
    for _, t in ipairs(tables) do
        if type(t) == "table" and t[name] ~= nil then
            return t[name]
        end
    end

    return nil
end

local function getVehicleTypeChainForVehicle(vehicle)
    local typeName = vehicle ~= nil and vehicle.typeName or nil
    local cacheKey = tostring(typeName or "")
    if cacheKey ~= "" then
        local cached = BMP_ExhaustPlusSpec._typeChainCache[cacheKey]
        if cached ~= nil then
            return cached
        end
    end

    local chain, seen = {}, {}

    pushUnique(chain, seen, typeName)
    pushUnique(chain, seen, getShortTypeName(typeName))

    local td = nil
    if vehicle ~= nil then
        td = vehicle.typeDesc or vehicle.vehicleType or vehicle.vehicleTypeDesc or vehicle.typeDefinition
    end
    if type(td) == "table" then
        buildChainFromTypeDesc(td, chain, seen)
    end

    if vehicle ~= nil and vehicle.getVehicleType ~= nil then
        local ok, vtObj = pcall(vehicle.getVehicleType, vehicle)
        if ok and type(vtObj) == "table" then
            buildChainFromTypeDesc(vtObj, chain, seen)
        end
    end

    local vtm = g_vehicleTypeManager
    if vtm ~= nil then
        local curName = typeName
        local guard = 0

        while curName ~= nil and curName ~= "" and guard < 64 do
            local vt = vtmLookupAny(vtm, curName)
            if vt == nil then
                local short = getShortTypeName(curName)
                if short ~= nil and short ~= curName then
                    vt = vtmLookupAny(vtm, short)
                    if vt ~= nil then
                        curName = short
                    end
                end
            end

            if vt == nil then
                break
            end

            local vn = getNameFromTypeDesc(vt)
            pushUnique(chain, seen, vn)
            pushUnique(chain, seen, getShortTypeName(vn))

            local parentName = getParentNameFromTypeDesc(vt)
            if parentName == nil then
                break
            end

            pushUnique(chain, seen, parentName)
            pushUnique(chain, seen, getShortTypeName(parentName))

            curName = parentName
            guard = guard + 1
        end
    end

    if cacheKey ~= "" then
        BMP_ExhaustPlusSpec._typeChainCache[cacheKey] = chain
    end
    return chain
end

local function resolvePresetKeyForVehicle(vehicle, spec)
    local ov = spec ~= nil and spec._presetKeyOverride or nil
    if ov ~= nil and ov ~= "" then
        if BMP_ExhaustPlusSpec.PARTICLE_PRESETS[ov] ~= nil then
            return ov
        else
            log("XML preset override '%s' not found in PARTICLE_PRESETS -> ignore", tostring(ov))
        end
    end

    local chain = getVehicleTypeChainForVehicle(vehicle)
    for _, n in ipairs(chain) do
        local key = BMP_ExhaustPlusSpec.TYPE_TO_PRESET[n]
        if key ~= nil and BMP_ExhaustPlusSpec.PARTICLE_PRESETS[key] ~= nil then
            return key
        end
    end

    local def = BMP_ExhaustPlusSpec.DEFAULT_PRESET_KEY or "default"
    if BMP_ExhaustPlusSpec.PARTICLE_PRESETS[def] ~= nil then
        return def
    end

    for k, _ in pairs(BMP_ExhaustPlusSpec.PARTICLE_PRESETS) do
        return k
    end
    return "default"
end

local function getPresetByKey(presetKey)
    local p = presetKey ~= nil and BMP_ExhaustPlusSpec.PARTICLE_PRESETS[presetKey] or nil
    if p == nil then
        p = BMP_ExhaustPlusSpec.PARTICLE_PRESETS[BMP_ExhaustPlusSpec.DEFAULT_PRESET_KEY or "default"]
    end
    return p
end

local function isNodeVisibleInHierarchy(node)
    if node == nil or node == 0 or not entityExists(node) then
        return false
    end

    local n = node
    local guard = 0

    while n ~= nil and n ~= 0 and entityExists(n) do
        if not getVisibility(n) then
            return false
        end

        local p = getParent(n)
        if p == nil or p == 0 then
            break
        end

        n = p
        guard = guard + 1
        if guard > 128 then
            break
        end
    end

    return true
end

local function pickActiveExhaustEntry(vehicle, spec)
    if spec == nil or spec.entries == nil or #spec.entries == 0 then
        return nil
    end

    local ov = spec._anchorNodeOverride
    if ov ~= nil and ov ~= 0 and entityExists(ov) and isNodeVisibleInHierarchy(ov) then
        local e = spec.byNode ~= nil and spec.byNode[ov] or nil
        if e ~= nil then
            return e
        end
    end

    for _, e in ipairs(spec.entries) do
        if e ~= nil and e.linkNode ~= nil and e.linkNode ~= 0 and entityExists(e.linkNode) then
            if isNodeVisibleInHierarchy(e.linkNode) then
                return e
            end
        end
    end

    return spec.entries[1]
end

local function stopEntryParticles(entry)
    if entry == nil then
        return
    end

    if entry.layers ~= nil then
        for _, L in ipairs(entry.layers) do
            if L ~= nil and L.ps ~= nil then
                if L.ps.isActive then
                    L.ps.isActive = false
                    ParticleUtil.setEmittingState(L.ps, false)
                end
                L.lastIntensity = 0
            end
        end
    end

    if entry.startLayer ~= nil and entry.startLayer.ps ~= nil then
        if entry.startLayer.ps.isActive then
            entry.startLayer.ps.isActive = false
            ParticleUtil.setEmittingState(entry.startLayer.ps, false)
        end
        entry.startLayer.lastIntensity = 0
    end

    entry.startPuffMs = 0
    entry.startPuffDelayMs = 0
    entry.startPuffArmed = false
end

local function stopAllEntries(spec)
    if spec == nil or spec.entries == nil then
        return
    end
    for _, e in ipairs(spec.entries) do
        stopEntryParticles(e)
        if e ~= nil then
            e.tuneTimerMs = 0
            e.resetCooldownMs = 0
            e.stage = 1
        end
    end
end

-- ---------------------------------------------------------
-- Specialization API
-- ---------------------------------------------------------
function BMP_ExhaustPlusSpec.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Motorized, specializations)
end

function BMP_ExhaustPlusSpec.initSpecialization()
    print("[BMP_ExhaustPlus] BMP_ExhaustPlusSpec initialized")

    local schema = Vehicle.xmlSchema
    schema:setXMLSpecializationType("BMP_ExhaustPlusSpec")

    local p1 = "vehicle.motorized.exhaustEffects.newExhaustEffect"
    schema:register(XMLValueType.NODE_INDEX, p1 .. "#node", "ExhaustPlus: override anchor node (node index string)")
    schema:register(XMLValueType.NODE_INDEX, p1 .. "(?)#node", "ExhaustPlus: override anchor node (node index string)")

    local p2 = "vehicle.exhaustEffects.newExhaustEffect"
    schema:register(XMLValueType.NODE_INDEX, p2 .. "#node", "ExhaustPlus: override anchor node (node index string)")
    schema:register(XMLValueType.NODE_INDEX, p2 .. "(?)#node", "ExhaustPlus: override anchor node (node index string)")

    local ppA = "vehicle.exhaustPlus"
    local ppB = "vehicle.motorized.exhaustEffects.exhaustPlus"
    schema:register(XMLValueType.STRING, ppA .. "#preset", "ExhaustPlus: particle preset key override")
    schema:register(XMLValueType.STRING, ppB .. "#preset", "ExhaustPlus: particle preset key override")

    local schemaSave = Vehicle.xmlSchemaSavegame
    local baseSavePath = string.format("vehicles.vehicle(?).%s.bmpExhaustPlus", BMP_ExhaustPlusSpec.modName)
    schemaSave:register(XMLValueType.BOOL, baseSavePath .. "#enabled", "ExhaustPlus: enabled on this vehicle", true)
    schemaSave:register(XMLValueType.STRING, baseSavePath .. "#preset", "ExhaustPlus: user preset override key", nil)

    schema:setXMLSpecializationType()
end

function BMP_ExhaustPlusSpec.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", BMP_ExhaustPlusSpec)
    SpecializationUtil.registerEventListener(vehicleType, "onDelete", BMP_ExhaustPlusSpec)
    SpecializationUtil.registerEventListener(vehicleType, "onUpdate", BMP_ExhaustPlusSpec)
    SpecializationUtil.registerEventListener(vehicleType, "onDraw", BMP_ExhaustPlusSpec) -- HUD текст
    SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", BMP_ExhaustPlusSpec)
    SpecializationUtil.registerEventListener(vehicleType, "saveToXMLFile", BMP_ExhaustPlusSpec)
end

function BMP_ExhaustPlusSpec.registerOverwrittenFunctions(vehicleType)
    SpecializationUtil.registerOverwrittenFunction(vehicleType, "onExhaustEffectI3DLoaded", BMP_ExhaustPlusSpec.onExhaustEffectI3DLoaded)
end

-- ---------------------------------------------------------
BMP_ExhaustPlusSpec._extraPSByPreset = BMP_ExhaustPlusSpec._extraPSByPreset or {}

local function resolveI3D(p)
    if p == nil then
        return nil
    end
    if p:sub(1, 6) == "$data/" then
        return p
    end
    return Utils.getFilename(p, BMP_ExhaustPlusSpec.modDirectory)
end

local function getOrCreatePresetCache(presetKey)
    local t = BMP_ExhaustPlusSpec._extraPSByPreset[presetKey]
    if t == nil then
        t = {
            emitterShape = nil,
            layerA = { sourceRoot = nil, sourceNode = nil, i3d = nil },
            layerB = { sourceRoot = nil, sourceNode = nil, i3d = nil },
            layerC = { sourceRoot = nil, sourceNode = nil, i3d = nil },
            layerD = { sourceRoot = nil, sourceNode = nil, i3d = nil },
            layerStart = { sourceRoot = nil, sourceNode = nil, i3d = nil }
        }
        BMP_ExhaustPlusSpec._extraPSByPreset[presetKey] = t
    end
    return t
end

local function loadEmitterShapeIfNeeded(cache)
    if cache.emitterShape ~= nil and cache.emitterShape ~= 0 and entityExists(cache.emitterShape) then
        return true
    end

    local shapePath = resolveI3D(BMP_ExhaustPlusSpec.extraEmitterShapeI3D)
    local shapeRoot = loadSharedI3DFile(shapePath, false, false)
    if shapeRoot == nil or shapeRoot == 0 then
        print(string.format("[BMP_ExhaustPlus] ExtraPS: failed to load emitterShape '%s'", tostring(shapePath)))
        return false
    end

    local refShape = getChildAt(shapeRoot, 0)
    if refShape == nil or refShape == 0 then
        print(string.format("[BMP_ExhaustPlus] ExtraPS: emitterShape child 0 missing '%s'", tostring(shapePath)))
        mpSafeDelete(shapeRoot)
        return false
    end

    link(getRootNode(), refShape)
    setClipDistance(refShape, BMP_ExhaustPlusSpec.particleClipDist or 150)
    cache.emitterShape = refShape
    return true
end

local function loadParticleSourceIfNeeded(layerCache, particlesI3D)
    if layerCache.sourceNode ~= nil and layerCache.sourceNode ~= 0 and entityExists(layerCache.sourceNode) then
        return true
    end

    local psFile = resolveI3D(particlesI3D)
    if psFile == nil or psFile == "" then
        return false
    end

    local root = loadI3DFile(psFile, false, false, false)
    if root == nil or root == 0 then
        print(string.format("[BMP_ExhaustPlus] ExtraPS: failed to load '%s'", tostring(psFile)))
        return false
    end

    local psNode = I3DUtil.indexToObject(root, BMP_ExhaustPlusSpec.particleNodeIndex or "0")
    if psNode == nil or psNode == 0 then
        print(string.format("[BMP_ExhaustPlus] ExtraPS: node '%s' not found in '%s'",
            tostring(BMP_ExhaustPlusSpec.particleNodeIndex), tostring(psFile)))
        mpSafeDelete(root)
        return false
    end

    layerCache.sourceRoot = root
    layerCache.sourceNode = psNode
    layerCache.i3d = psFile
    return true
end

function BMP_ExhaustPlusSpec.loadExtraParticleReferencesForPreset(presetKey, preset)
    if not BMP_ExhaustPlusSpec.extraParticlesEnable then
        return false
    end

    if presetKey == nil or presetKey == "" then
        presetKey = BMP_ExhaustPlusSpec.DEFAULT_PRESET_KEY or "default"
    end
    if preset == nil then
        preset = getPresetByKey(presetKey)
    end
    if preset == nil then
        return false
    end

    local cache = getOrCreatePresetCache(presetKey)

    if not loadEmitterShapeIfNeeded(cache) then
        return false
    end

    local okA, okB, okC, okD, okS = true, true, true, true, true

    if preset.layerA_enable then
        okA = loadParticleSourceIfNeeded(cache.layerA, preset.layerA_particlesI3D)
    end
    if preset.layerB_enable then
        okB = loadParticleSourceIfNeeded(cache.layerB, preset.layerB_particlesI3D)
    end
    if preset.layerC_enable then
        okC = loadParticleSourceIfNeeded(cache.layerC, preset.layerC_particlesI3D)
    end
    if preset.layerD_enable then
        okD = loadParticleSourceIfNeeded(cache.layerD, preset.layerD_particlesI3D)
    end

    if preset.layerStart_enable then
        okS = loadParticleSourceIfNeeded(cache.layerStart, preset.layerStart_particlesI3D)
    end

    if (not okA) and (not okB) and (not okC) and (not okD) and (not okS) then
        return false
    end

    log("ExtraPS refs loaded preset='%s': emitterShape='%s' A='%s' B='%s' C='%s' D='%s' S='%s'",
        tostring(presetKey),
        tostring(BMP_ExhaustPlusSpec.extraEmitterShapeI3D),
        tostring(cache.layerA.i3d),
        tostring(cache.layerB.i3d),
        tostring(cache.layerC.i3d),
        tostring(cache.layerD.i3d),
        tostring(cache.layerStart.i3d))

    return true
end

-- ---------------------------------------------------------
-- Anchor override
-- ---------------------------------------------------------
local function tryResolveNewExhaustNode(vehicle)
    if vehicle == nil or vehicle.xmlFile == nil then
        return nil
    end

    local comps = vehicle.components
    local maps  = vehicle.i3dMappings

    if comps == nil then
        log("newExhaustEffect: vehicle.components is nil -> fallback to vanilla anchor")
        return nil
    end

    local keys = {
        "vehicle.motorized.exhaustEffects.newExhaustEffect#node",
        "vehicle.motorized.exhaustEffects.newExhaustEffect(?)#node",
        "vehicle.exhaustEffects.newExhaustEffect#node",
        "vehicle.exhaustEffects.newExhaustEffect(?)#node"
    }

    for _, key in ipairs(keys) do
        if vehicle.xmlFile:hasProperty(key) then
            local node = vehicle.xmlFile:getValue(key, nil, comps, maps)
            if node ~= nil and node ~= 0 and entityExists(node) then
                log("Using newExhaustEffect anchor '%s' -> nodeId=%s", key, tostring(node))
                return node
            else
                log("newExhaustEffect found at '%s' but could not be resolved (value invalid) -> continue", key)
            end
        end
    end

    return nil
end

local function tryResolvePresetOverride(vehicle)
    if vehicle == nil or vehicle.xmlFile == nil then
        return nil
    end

    local k1 = "vehicle.exhaustPlus#preset"
    local k2 = "vehicle.motorized.exhaustEffects.exhaustPlus#preset"

    local p = nil
    if vehicle.xmlFile:hasProperty(k2) then
        p = vehicle.xmlFile:getValue(k2, nil)
    end
    if (p == nil or p == "") and vehicle.xmlFile:hasProperty(k1) then
        p = vehicle.xmlFile:getValue(k1, nil)
    end

    if p ~= nil and p ~= "" then
        return p
    end
    return nil
end

-- ---------------------------------------------------------
local function captureBaseParams(psTbl)
    ParticleUtil.setMaxNumOfParticlesToEmitScale(psTbl, 1.0)
    psTbl._bmpBase = {
        speed     = ParticleUtil.getParticleSystemSpeed(psTbl) or 0,
        speedRand = ParticleUtil.getParticleSystemSpeedRandom(psTbl) or 0,
        tangent   = ParticleUtil.getParticleSystemTangentSpeed(psTbl) or 0,
        normal    = ParticleUtil.getParticleSystemNormalSpeed(psTbl) or 0
    }
end

local function applyIntensityTuning(ps, intensity, emitMin, emitMax, spMin, spMax)
    if ps == nil or not ps.isValid then
        return
    end

    if ps._bmpBase == nil then
        captureBaseParams(ps)
    end

    local base = ps._bmpBase
    if base == nil then
        return
    end

    local emitScale = lerp(emitMin, emitMax, intensity)
    ParticleUtil.setMaxNumOfParticlesToEmitScale(ps, emitScale)

    local spMul = lerp(spMin, spMax, intensity)

    if base.speed ~= nil then
        ParticleUtil.setParticleSystemSpeed(ps, base.speed * spMul)
    end
    if base.speedRand ~= nil then
        ParticleUtil.setParticleSystemSpeedRandom(ps, base.speedRand * spMul)
    end
    if base.tangent ~= nil then
        ParticleUtil.setParticleSystemTangentSpeed(ps, base.tangent * spMul)
    end
    if base.normal ~= nil then
        ParticleUtil.setParticleSystemNormalSpeed(ps, base.normal * spMul)
    end
end

-- ---------------------------------------------------------
local function createLayerInstance(vehicle, emitterTG, emitterShape, layerCache, layerTag)
    if layerCache == nil or layerCache.sourceNode == nil or layerCache.sourceNode == 0 or not entityExists(layerCache.sourceNode) then
        return nil
    end

    local psClone = clone(layerCache.sourceNode, true, false, true)
    link(emitterTG, psClone)

    local psTbl = {}
    ParticleUtil.loadParticleSystemFromNode(psClone, psTbl, false, true, false)
    ParticleUtil.setEmitterShape(psTbl, emitterShape)
    ParticleUtil.setEmittingState(psTbl, false)
    psTbl.isActive = false
    psTbl._bmpBase = nil

    return {
        tag = layerTag,
        psClone = psClone,
        ps = psTbl,
        lastIntensity = 0
    }
end

local function createEntryOnLinkNode(vehicle, linkNode, presetKey, preset)
    local cache = getOrCreatePresetCache(presetKey or (BMP_ExhaustPlusSpec.DEFAULT_PRESET_KEY or "default"))
    if cache == nil or cache.emitterShape == nil or cache.emitterShape == 0 or not entityExists(cache.emitterShape) then
        return nil
    end
    if linkNode == nil or linkNode == 0 or not entityExists(linkNode) then
        return nil
    end

    local offset = createTransformGroup("bmpExhaustOffset")
    link(linkNode, offset)
    setTranslation(offset, BMP_ExhaustPlusSpec.offsetX or 0, BMP_ExhaustPlusSpec.offsetY or 0, BMP_ExhaustPlusSpec.offsetZ or 0)
    setRotation(offset, 0, 0, 0)

    local emitter = createTransformGroup("bmpExhaustEmitter")
    link(offset, emitter)
    setTranslation(emitter, 0, 0, 0)
    setRotation(emitter, 0, 0, 0)

    local emitterShape = clone(cache.emitterShape, true, false, true)
    link(emitter, emitterShape)
    setClipDistance(emitterShape, BMP_ExhaustPlusSpec.particleClipDist or 150)

    local layers = {}
    local startLayer = nil

    if preset.layerA_enable then
        local L = createLayerInstance(vehicle, emitter, emitterShape, cache.layerA, "A")
        if L ~= nil then table.insert(layers, L) end
    end
    if preset.layerB_enable then
        local L = createLayerInstance(vehicle, emitter, emitterShape, cache.layerB, "B")
        if L ~= nil then table.insert(layers, L) end
    end
    if preset.layerC_enable then
        local L = createLayerInstance(vehicle, emitter, emitterShape, cache.layerC, "C")
        if L ~= nil then table.insert(layers, L) end
    end
    if preset.layerD_enable then
        local L = createLayerInstance(vehicle, emitter, emitterShape, cache.layerD, "D")
        if L ~= nil then table.insert(layers, L) end
    end

    if preset.layerStart_enable then
        local Ls = createLayerInstance(vehicle, emitter, emitterShape, cache.layerStart, "S")
        startLayer = Ls
    end

    if #layers == 0 and startLayer == nil then
        mpSafeDelete(emitterShape)
        mpSafeDelete(emitter)
        mpSafeDelete(offset)
        return nil
    end

    return {
        linkNode = linkNode,
        offset = offset,
        emitter = emitter,
        emitterShape = emitterShape,

        layers = layers,
        startLayer = startLayer,

        tuneTimerMs = 0,
        resetCooldownMs = 0,

        stage = 1,

        startPuffMs = 0,
        startPuffDelayMs = 0,
        startPuffArmed = false
    }
end

-- =========================================================
-- FIX #3: onExhaustEffectI3DLoaded
-- =========================================================
function BMP_ExhaustPlusSpec:onExhaustEffectI3DLoaded(superFunc, i3dNode, failedReason, args)
    superFunc(self, i3dNode, failedReason, args)

    if not self.isClient or not BMP_ExhaustPlusSpec.extraParticlesEnable then
        return
    end
    if args == nil or args.linkNode == nil then
        return
    end

    local spec = self.spec_bmpExhaustPlus
    if spec == nil then
        spec = {}
        self.spec_bmpExhaustPlus = spec
    end

    spec.entries = spec.entries or {}
    spec.byNode  = spec.byNode  or {}

    if spec._presetKey == nil then
        spec._presetKey = resolvePresetKeyForVehicle(self, spec)
    end
    spec._preset = spec._preset or getPresetByKey(spec._presetKey)
    if spec._preset == nil then
        return
    end

    if not BMP_ExhaustPlusSpec.loadExtraParticleReferencesForPreset(spec._presetKey, spec._preset) then
        return
    end

    local ln = args.linkNode
    if ln == nil or ln == 0 or not entityExists(ln) then
        return
    end

    if spec.byNode[ln] == nil then
        local e = createEntryOnLinkNode(self, ln, spec._presetKey, spec._preset)
        if e ~= nil then
            table.insert(spec.entries, e)
            spec.byNode[ln] = e
            log("Attached extra exhaust entry to linkNode=%s preset='%s' type='%s' vehicle='%s'",
                tostring(ln), tostring(spec._presetKey), tostring(self.typeName),
                tostring(self.typeName or self.configFileName or "vehicle"))
        end
    end

    if (spec.userEnabled ~= false) and (BMP_ExhaustPlusSpec.HIDE_VANILLA_EXHAUST or spec._vanillaExhaustHideRequested) then
        BMP_ExhaustPlusSpec.setVanillaExhaustEnabled(self, false)
    end
end

-- =========================================================
-- onLoad
-- =========================================================
function BMP_ExhaustPlusSpec:onLoad(savegame)
    if not self.isClient then
        return
    end

    local spec = self.spec_bmpExhaustPlus
    if spec == nil then
        spec = {}
        self.spec_bmpExhaustPlus = spec
    end

    spec.entries = spec.entries or {}
    spec.byNode  = spec.byNode  or {}

    spec.userEnabled = true
    spec.userPresetKey = nil
    if savegame ~= nil and savegame.xmlFile ~= nil and savegame.key ~= nil then
        local xmlFile = savegame.xmlFile
        local skey = BMP_ExhaustPlusSpec.getSaveKeyForVehicle(savegame.key)
        spec.userEnabled = xmlFile:getValue(skey .. "#enabled", true)

        local p = xmlFile:getValue(skey .. "#preset", nil)
        if p ~= nil and p ~= "" and BMP_ExhaustPlusSpec.PARTICLE_PRESETS[p] ~= nil then
            spec.userPresetKey = p
        end
    end

    spec._anchorNodeOverride = tryResolveNewExhaustNode(self)
    spec._presetKeyOverride  = tryResolvePresetOverride(self)

    if spec.userPresetKey ~= nil then
        spec._presetKeyOverride = spec.userPresetKey
    end

    spec._presetKey = resolvePresetKeyForVehicle(self, spec)
    spec._preset    = getPresetByKey(spec._presetKey)

    if spec._preset == nil then
        log("No preset resolved for type '%s' -> ExhaustPlus disabled for this vehicle", tostring(self.typeName))
        return
    end

    BMP_ExhaustPlusSpec.loadExtraParticleReferencesForPreset(spec._presetKey, spec._preset)

    spec._activeEntry    = nil
    spec._activeLinkNode = nil
    spec._activeScanMs   = 0

    spec._vanillaExhaustHideRequested = (BMP_ExhaustPlusSpec.HIDE_VANILLA_EXHAUST == true) and (spec.userEnabled ~= false)
    spec._vanillaExhaustDisabled = false

    spec._wasRunning = false

    spec._hudStatusText = nil
    spec._hudStatusMs   = 0

    local chain = getVehicleTypeChainForVehicle(self)
    log("Preset resolved: type='%s' chain=%s -> preset='%s' userEnabled=%s",
        tostring(self.typeName),
        (#chain > 0 and table.concat(chain, " > ") or "nil"),
        tostring(spec._presetKey),
        tostring(spec.userEnabled ~= false))
end

function BMP_ExhaustPlusSpec:onDelete()
    local spec = self.spec_bmpExhaustPlus

    if spec ~= nil and spec.entries ~= nil then
        for _, e in ipairs(spec.entries) do
            if e ~= nil and e.layers ~= nil then
                for _, L in ipairs(e.layers) do
                    if L ~= nil and L.ps ~= nil then
                        ParticleUtil.setEmittingState(L.ps, false)
                    end
                    if L ~= nil and L.psClone ~= nil then
                        mpSafeDelete(L.psClone)
                    end
                end
            end

            if e ~= nil and e.startLayer ~= nil then
                if e.startLayer.ps ~= nil then
                    ParticleUtil.setEmittingState(e.startLayer.ps, false)
                end
                if e.startLayer.psClone ~= nil then
                    mpSafeDelete(e.startLayer.psClone)
                end
            end

            if e ~= nil and e.emitterShape ~= nil then mpSafeDelete(e.emitterShape) end
            if e ~= nil and e.emitter ~= nil then mpSafeDelete(e.emitter) end
            if e ~= nil and e.offset ~= nil then mpSafeDelete(e.offset) end
        end

        spec.entries = nil
        spec.byNode  = nil
        spec._activeEntry = nil
        spec._activeLinkNode = nil
    end
end

-- HUD draw event
function BMP_ExhaustPlusSpec:onDraw()
    if not self.isClient then return end
    local spec = self.spec_bmpExhaustPlus
    if spec == nil then return end
    drawHud(spec)
end

local function applyLayerTuningByTag(L, intensity)
    if L.tag == "A" then
        applyIntensityTuning(L.ps, intensity,
            BMP_ExhaustPlusSpec.A_EMIT_SCALE_MIN, BMP_ExhaustPlusSpec.A_EMIT_SCALE_MAX,
            BMP_ExhaustPlusSpec.A_SPEED_MUL_MIN,  BMP_ExhaustPlusSpec.A_SPEED_MUL_MAX)
    elseif L.tag == "B" then
        applyIntensityTuning(L.ps, intensity,
            BMP_ExhaustPlusSpec.B_EMIT_SCALE_MIN, BMP_ExhaustPlusSpec.B_EMIT_SCALE_MAX,
            BMP_ExhaustPlusSpec.B_SPEED_MUL_MIN,  BMP_ExhaustPlusSpec.B_SPEED_MUL_MAX)
    elseif L.tag == "C" then
        applyIntensityTuning(L.ps, intensity,
            BMP_ExhaustPlusSpec.C_EMIT_SCALE_MIN, BMP_ExhaustPlusSpec.C_EMIT_SCALE_MAX,
            BMP_ExhaustPlusSpec.C_SPEED_MUL_MIN,  BMP_ExhaustPlusSpec.C_SPEED_MUL_MAX)
    elseif L.tag == "D" then
        applyIntensityTuning(L.ps, intensity,
            BMP_ExhaustPlusSpec.D_EMIT_SCALE_MIN, BMP_ExhaustPlusSpec.D_EMIT_SCALE_MAX,
            BMP_ExhaustPlusSpec.D_SPEED_MUL_MIN,  BMP_ExhaustPlusSpec.D_SPEED_MUL_MAX)
    end
end

local function stageToTag(stage)
    if stage == 4 then return "D" end
    if stage == 3 then return "C" end
    if stage == 2 then return "B" end
    return "A"
end

local function updateStageExclusive(entry, intensity)
    local st = entry.stage or 1

    local abOn  = BMP_ExhaustPlusSpec.THR_AB_ON
    local bcOn  = BMP_ExhaustPlusSpec.THR_BC_ON
    local cdOn  = BMP_ExhaustPlusSpec.THR_CD_ON

    local abOff = BMP_ExhaustPlusSpec.THR_AB_OFF
    local bcOff = BMP_ExhaustPlusSpec.THR_BC_OFF
    local cdOff = BMP_ExhaustPlusSpec.THR_CD_OFF

    if st == 1 and intensity >= abOn then
        st = 2
    elseif st == 2 and intensity >= bcOn then
        st = 3
    elseif st == 3 and intensity >= cdOn then
        st = 4
    end

    if st == 4 and intensity < cdOff then
        st = 3
    elseif st == 3 and intensity < bcOff then
        st = 2
    elseif st == 2 and intensity < abOff then
        st = 1
    end

    entry.stage = st
end

-- =========================================================
-- onUpdate
-- =========================================================
function BMP_ExhaustPlusSpec:onUpdate(dt)
    if not self.isClient then
        return
    end

    local spec = self.spec_bmpExhaustPlus
    if spec == nil then
        return
    end

    tickHud(spec, dt)

    local specM = self.spec_motorized
    if specM == nil then
        return
    end

    if spec.userEnabled == false then
        BMP_ExhaustPlusSpec.stopAllCustomExhaust(spec)
        BMP_ExhaustPlusSpec.setVanillaExhaustEnabled(self, true)
        spec._wasRunning = false
        return
    end

    if BMP_ExhaustPlusSpec.HIDE_VANILLA_EXHAUST then
        BMP_ExhaustPlusSpec.setVanillaExhaustEnabled(self, false)
    end

    if spec.entries == nil or #spec.entries == 0 then
        spec._wasRunning = false
        return
    end

    local motorState = self:getMotorState()
    local isRunning = (motorState == MotorState.STARTING or motorState == MotorState.ON)

    if not isRunning then
        stopAllEntries(spec)
        spec._wasRunning = false
        return
    end

    spec._activeScanMs = (spec._activeScanMs or 0) + dt
    local scanInt = BMP_ExhaustPlusSpec.ACTIVE_SCAN_INTERVAL_MS or 200

    if spec._activeEntry == nil or spec._activeScanMs >= scanInt then
        spec._activeScanMs = 0

        local newActive = pickActiveExhaustEntry(self, spec)
        if newActive ~= spec._activeEntry then
            stopAllEntries(spec)

            spec._activeEntry = newActive
            spec._activeLinkNode = newActive ~= nil and newActive.linkNode or nil

            log("Active exhaust switched to linkNode=%s preset='%s' vehicle='%s'",
                tostring(spec._activeLinkNode), tostring(spec._presetKey),
                tostring(self.typeName or self.configFileName or "vehicle"))
        end
    end

    local e = spec._activeEntry
    if e == nil then
        spec._wasRunning = isRunning
        return
    end

    local justStarted = (spec._wasRunning ~= true) and (isRunning == true)
    if justStarted and e.startLayer ~= nil and e.startLayer.ps ~= nil then
        e.startPuffDelayMs = BMP_ExhaustPlusSpec.START_PUFF_DELAY_MS or 0
        e.startPuffMs = BMP_ExhaustPlusSpec.START_PUFF_DURATION_MS or 1000
        e.startPuffArmed = true

        log("Start puff armed: delay=%d ms, duration=%d ms preset='%s' vehicle='%s'",
            e.startPuffDelayMs or 0, e.startPuffMs or 0, tostring(spec._presetKey), tostring(self.typeName or "vehicle"))
    end

    spec._wasRunning = isRunning

    if e.startPuffArmed == true and (e.startPuffMs or 0) > 0 then
        if (e.startPuffDelayMs or 0) > 0 then
            e.startPuffDelayMs = math.max((e.startPuffDelayMs or 0) - dt, 0)
        end

        if (e.startPuffDelayMs or 0) <= 0 then
            e.startPuffArmed = false

            if e.startLayer ~= nil and e.startLayer.ps ~= nil then
                ParticleUtil.resetNumOfEmittedParticles(e.startLayer.ps)
            end
            e.resetCooldownMs = BMP_ExhaustPlusSpec.RESET_COOLDOWN_MS or 650

            log("Start puff triggered after delay (%d ms) preset='%s' vehicle='%s'",
                (BMP_ExhaustPlusSpec.START_PUFF_DELAY_MS or 0),
                tostring(spec._presetKey), tostring(self.typeName or "vehicle"))
        end
    end

    if (e.startPuffMs or 0) > 0 and (e.startPuffDelayMs or 0) <= 0 and (e.startPuffArmed ~= true) then
        e.startPuffMs = math.max((e.startPuffMs or 0) - dt, 0)

        if e.layers ~= nil then
            for _, L in ipairs(e.layers) do
                if L ~= nil and L.ps ~= nil and L.ps.isActive then
                    L.ps.isActive = false
                    ParticleUtil.setEmittingState(L.ps, false)
                    L.lastIntensity = 0
                end
            end
        end

        local S = e.startLayer
        if S ~= nil and S.ps ~= nil then
            if S.ps.isActive ~= true then
                S.ps.isActive = true
                ParticleUtil.setEmittingState(S.ps, true)
                ParticleUtil.resetNumOfEmittedParticles(S.ps)
            end

            applyIntensityTuning(S.ps, 1.0,
                BMP_ExhaustPlusSpec.D_EMIT_SCALE_MIN, BMP_ExhaustPlusSpec.D_EMIT_SCALE_MAX,
                BMP_ExhaustPlusSpec.D_SPEED_MUL_MIN,  BMP_ExhaustPlusSpec.D_SPEED_MUL_MAX)
        end

        if (e.startPuffMs or 0) <= 0 then
            if S ~= nil and S.ps ~= nil and S.ps.isActive then
                S.ps.isActive = false
                ParticleUtil.setEmittingState(S.ps, false)
                S.lastIntensity = 0
            end
        end

        return
    end

    local load = clamp(specM.actualLoadPercentage or 0, 0, 1)

    local rpmN = 0
    if specM.motor ~= nil and specM.motor.getMaxRpm ~= nil then
        local maxRpm = specM.motor:getMaxRpm()
        if maxRpm ~= nil and maxRpm > 0 then
            rpmN = clamp((self:getMotorRpmReal() or 0) / maxRpm, 0, 1)
        end
    end

    local rpmGate = smoothstep(BMP_ExhaustPlusSpec.RPM_GATE_MIN, BMP_ExhaustPlusSpec.RPM_GATE_MAX, rpmN)
    local intensity = (load ^ (BMP_ExhaustPlusSpec.LOAD_EXP or 1.0)) * (rpmN ^ (BMP_ExhaustPlusSpec.RPM_EXP or 1.0)) * rpmGate
    intensity = clamp(intensity, 0, 1)

    local intensityA = math.max(intensity, BMP_ExhaustPlusSpec.A_IDLE_INTENSITY_FLOOR or 0)
    local emitAny = (intensityA >= (BMP_ExhaustPlusSpec.MIN_INTENSITY_TO_EMIT or 0.01))

    updateStageExclusive(e, intensity)
    local activeTag = stageToTag(e.stage or 1)

    if (e.resetCooldownMs or 0) > 0 then
        e.resetCooldownMs = math.max((e.resetCooldownMs or 0) - dt, 0)
    end

    e.tuneTimerMs = (e.tuneTimerMs or 0) + dt
    local doTune = false
    if e.tuneTimerMs >= (BMP_ExhaustPlusSpec.TUNE_INTERVAL_MS or 80) then
        e.tuneTimerMs = 0
        doTune = true
    end

    if e.layers ~= nil then
        for _, L in ipairs(e.layers) do
            if L ~= nil and L.ps ~= nil then
                local shouldEmit = emitAny and (L.tag == activeTag)

                if L.ps.isActive ~= shouldEmit then
                    L.ps.isActive = shouldEmit
                    ParticleUtil.setEmittingState(L.ps, shouldEmit)

                    if shouldEmit then
                        ParticleUtil.resetNumOfEmittedParticles(L.ps)
                        e.resetCooldownMs = BMP_ExhaustPlusSpec.RESET_COOLDOWN_MS or 650
                    else
                        L.lastIntensity = 0
                    end
                end

                if shouldEmit and doTune then
                    local useI = 0

                    local abOn = BMP_ExhaustPlusSpec.THR_AB_ON
                    local bcOn = BMP_ExhaustPlusSpec.THR_BC_ON
                    local cdOn = BMP_ExhaustPlusSpec.THR_CD_ON

                    if L.tag == "A" then
                        useI = intensityA
                    elseif L.tag == "B" then
                        useI = clamp((intensity - abOn) / math.max(bcOn - abOn, 0.001), 0, 1)
                    elseif L.tag == "C" then
                        useI = clamp((intensity - bcOn) / math.max(cdOn - bcOn, 0.001), 0, 1)
                    elseif L.tag == "D" then
                        useI = clamp((intensity - cdOn) / math.max(1 - cdOn, 0.001), 0, 1)
                    end

                    local delta = useI - (L.lastIntensity or 0)

                    if delta >= (BMP_ExhaustPlusSpec.RESET_ON_RAMP_DELTA or 0.18) and (e.resetCooldownMs or 0) <= 0 then
                        ParticleUtil.resetNumOfEmittedParticles(L.ps)
                        e.resetCooldownMs = BMP_ExhaustPlusSpec.RESET_COOLDOWN_MS or 650
                    end

                    if math.abs(delta) >= (BMP_ExhaustPlusSpec.TUNE_EPS or 0.02) then
                        L.lastIntensity = useI
                        applyLayerTuningByTag(L, useI)
                    end
                end
            end
        end
    end
end

-- =========================================================
-- Per-vehicle enable + preset override (saved in vehicles.xml)
-- =========================================================
function BMP_ExhaustPlusSpec.getSaveKeyForVehicle(keyOrBase)
    if keyOrBase == nil then
        return nil
    end

    local k = tostring(keyOrBase)
    local suffixFull = "." .. tostring(BMP_ExhaustPlusSpec.modName) .. ".bmpExhaustPlus"
    local suffixShort = ".bmpExhaustPlus"

    if k:sub(-#suffixFull) == suffixFull then
        return k
    end
    if k:sub(-#suffixShort) == suffixShort then
        return k
    end

    return k .. suffixFull
end

function BMP_ExhaustPlusSpec.getAllPresetKeysOrdered()
    local keys = { BMP_ExhaustPlusSpec.DEFAULT_PRESET_KEY or "default", "tractor", "tractor2", "tractorPuff", "combineDrivable", "baseDrivable", "carFillable" }
    local seen = {}
    local out = {}
    for _, k in ipairs(keys) do
        if k ~= nil and BMP_ExhaustPlusSpec.PARTICLE_PRESETS[k] ~= nil and not seen[k] then
            table.insert(out, k); seen[k] = true
        end
    end
    for k, _ in pairs(BMP_ExhaustPlusSpec.PARTICLE_PRESETS) do
        if not seen[k] then
            table.insert(out, k); seen[k] = true
        end
    end
    return out
end

-- =========================================================
-- FIX #1: setVanillaExhaustEnabled
-- =========================================================
function BMP_ExhaustPlusSpec.setVanillaExhaustEnabled(vehicle, enabled)
    local specM = vehicle ~= nil and vehicle.spec_motorized or nil
    if specM == nil then
        return false
    end

    local appliedAny = false

    if specM.exhaustParticleSystems ~= nil then
        for _, ps in pairs(specM.exhaustParticleSystems) do
            if ps ~= nil and ps.isValid ~= false then
                ParticleUtil.setEmittingState(ps, enabled)
                appliedAny = true
            end
        end
    end

    if specM.exhaustEffects ~= nil then
        for _, e in pairs(specM.exhaustEffects) do
            if e ~= nil then
                if e.effectNode ~= nil and e.effectNode ~= 0 and entityExists(e.effectNode) then
                    setVisibility(e.effectNode, enabled)
                    appliedAny = true
                end

                if e.particleSystems ~= nil then
                    for _, ps in pairs(e.particleSystems) do
                        if ps ~= nil and ps.isValid ~= false then
                            ParticleUtil.setEmittingState(ps, enabled)
                            appliedAny = true
                        end
                    end
                end
            end
        end
    end

    local spec = vehicle.spec_bmpExhaustPlus
    if spec ~= nil and appliedAny then
        spec._vanillaExhaustDisabled = not enabled
    end

    return appliedAny
end

function BMP_ExhaustPlusSpec.stopAllCustomExhaust(spec)
    if spec == nil or spec.entries == nil then
        return
    end
    for _, e in ipairs(spec.entries) do
        if e ~= nil and e.layers ~= nil then
            for _, L in ipairs(e.layers) do
                if L ~= nil and L.ps ~= nil then
                    ParticleUtil.setEmittingState(L.ps, false)
                end
            end
        end
        if e ~= nil and e.startLayer ~= nil and e.startLayer.ps ~= nil then
            ParticleUtil.setEmittingState(e.startLayer.ps, false)
        end
    end
end

function BMP_ExhaustPlusSpec.rebuildCustomEntriesForPreset(vehicle, newPresetKey)
    local spec = vehicle.spec_bmpExhaustPlus
    if spec == nil or spec.entries == nil then
        return
    end

    local linkNodes = {}
    for _, e in ipairs(spec.entries) do
        if e ~= nil and e.linkNode ~= nil then
            table.insert(linkNodes, e.linkNode)
        end
    end

    for _, e in ipairs(spec.entries) do
        if e ~= nil and e.layers ~= nil then
            for _, L in ipairs(e.layers) do
                if L ~= nil and L.ps ~= nil then
                    ParticleUtil.setEmittingState(L.ps, false)
                end
                if L ~= nil and L.psClone ~= nil then
                    mpSafeDelete(L.psClone)
                end
            end
        end

        if e ~= nil and e.startLayer ~= nil then
            if e.startLayer.ps ~= nil then
                ParticleUtil.setEmittingState(e.startLayer.ps, false)
            end
            if e.startLayer.psClone ~= nil then
                mpSafeDelete(e.startLayer.psClone)
            end
        end

        if e ~= nil and e.emitterShape ~= nil then mpSafeDelete(e.emitterShape) end
        if e ~= nil and e.emitter ~= nil then mpSafeDelete(e.emitter) end
        if e ~= nil and e.offset ~= nil then mpSafeDelete(e.offset) end
    end

    spec.entries = {}
    spec.byNode  = {}
    spec._activeEntry = nil
    spec._activeLinkNode = nil

    spec._presetKey = newPresetKey
    spec._preset = getPresetByKey(newPresetKey)
    BMP_ExhaustPlusSpec.loadExtraParticleReferencesForPreset(spec._presetKey, spec._preset)

    for _, ln in ipairs(linkNodes) do
        if ln ~= nil and ln ~= 0 and entityExists(ln) then
            local e = createEntryOnLinkNode(vehicle, ln, spec._presetKey, spec._preset)
            if e ~= nil then
                table.insert(spec.entries, e)
                spec.byNode[ln] = e
            end
        end
    end
end

function BMP_ExhaustPlusSpec.updateActionTexts(vehicle)
    local spec = vehicle ~= nil and vehicle.spec_bmpExhaustPlus or nil
    if spec == nil then
        return
    end

    if g_inputBinding == nil or g_inputBinding.setActionEventText == nil then
        return
    end

    local enabled = (spec.userEnabled ~= false)

    local toggleText = "ExhaustPlus: Vanilla exhaust"
    local presetText = "ExhaustPlus: Next preset"
    if g_i18n ~= nil then
        toggleText = g_i18n:getText("action_bmp_exhaustplus_vanilla") or toggleText
        presetText = g_i18n:getText("action_bmp_exhaustplus_presetNext") or presetText
    end

    if spec._actionEventId_toggle ~= nil then
        local suffix = enabled and " [ON]" or " [OFF]"
        g_inputBinding:setActionEventText(spec._actionEventId_toggle, toggleText .. suffix)
    end

    if spec._actionEventId_preset ~= nil then
        local p = spec.userPresetKey or spec._presetKey or (BMP_ExhaustPlusSpec.DEFAULT_PRESET_KEY or "default")
        g_inputBinding:setActionEventText(spec._actionEventId_preset, presetText .. " [" .. tostring(p) .. "]")
    end
end

function BMP_ExhaustPlusSpec:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
    if not self.isClient then
        return
    end

    local spec = self.spec_bmpExhaustPlus
    if spec == nil then
        return
    end

    if spec.actionEvents ~= nil then
        self:clearActionEventsTable(spec.actionEvents)
    end
    spec.actionEvents = spec.actionEvents or {}

    if isActiveForInputIgnoreSelection then
        local _, id1 = self:addActionEvent(spec.actionEvents, InputAction.BMP_EXHAUSTPLUS_VANILLA, self, BMP_ExhaustPlusSpec.actionEventToggleEnabled, false, true, false, true)
        spec._actionEventId_toggle = id1

        local _, id2 = self:addActionEvent(spec.actionEvents, InputAction.BMP_EXHAUSTPLUS_PRESET_NEXT, self, BMP_ExhaustPlusSpec.actionEventPresetNext, false, true, false, true)
        spec._actionEventId_preset = id2

        BMP_ExhaustPlusSpec.updateActionTexts(self)
    end
end

function BMP_ExhaustPlusSpec.actionEventToggleEnabled(vehicle, actionName, inputValue, callbackState, isAnalog)
    local spec = vehicle.spec_bmpExhaustPlus
    if spec == nil then
        return
    end

    spec.userEnabled = not (spec.userEnabled ~= false)

    if spec.userEnabled then
        BMP_ExhaustPlusSpec.setVanillaExhaustEnabled(vehicle, false)
    else
        BMP_ExhaustPlusSpec.stopAllCustomExhaust(spec)
        BMP_ExhaustPlusSpec.setVanillaExhaustEnabled(vehicle, true)
    end

    BMP_ExhaustPlusSpec.updateActionTexts(vehicle)
    BMP_ExhaustPlusSpec.pushHudStatus(vehicle) -- HUD текст
end

function BMP_ExhaustPlusSpec.actionEventPresetNext(vehicle, actionName, inputValue, callbackState, isAnalog)
    local spec = vehicle.spec_bmpExhaustPlus
    if spec == nil then
        return
    end

    local keys = BMP_ExhaustPlusSpec.getAllPresetKeysOrdered()
    if #keys == 0 then
        return
    end

    local cur = spec.userPresetKey or spec._presetKey or (BMP_ExhaustPlusSpec.DEFAULT_PRESET_KEY or "default")
    local idx = 1
    for i, k in ipairs(keys) do
        if k == cur then
            idx = i
            break
        end
    end

    idx = idx + 1
    if idx > #keys then
        idx = 1
    end

    local nextKey = keys[idx]
    spec.userPresetKey = nextKey

    BMP_ExhaustPlusSpec.rebuildCustomEntriesForPreset(vehicle, nextKey)

    if spec.userEnabled ~= false then
        BMP_ExhaustPlusSpec.setVanillaExhaustEnabled(vehicle, false)
    end

    BMP_ExhaustPlusSpec.updateActionTexts(vehicle)
    BMP_ExhaustPlusSpec.pushHudStatus(vehicle) -- HUD текст
end

function BMP_ExhaustPlusSpec:saveToXMLFile(xmlFile, key)
    local spec = self.spec_bmpExhaustPlus
    if spec == nil or xmlFile == nil or key == nil then
        return
    end

    local saveKey = BMP_ExhaustPlusSpec.getSaveKeyForVehicle(key)

    xmlFile:setValue(saveKey .. "#enabled", spec.userEnabled ~= false)

    if spec.userPresetKey ~= nil and spec.userPresetKey ~= "" then
        xmlFile:setValue(saveKey .. "#preset", spec.userPresetKey)
    else
        xmlFile:removeProperty(saveKey .. "#preset")
    end
end
